<?php
/**
 * Plugin Name: Botón Flotante WhatsApp Personalizado
 * Description: Añade un botón flotante de WhatsApp en la esquina inferior derecha con un pop-up de múltiples opciones.
 * Version: 1.0.1
 * Author: SGC
 * License: GPL2
 */

if ( ! defined( 'ABSPATH' ) ) exit;

final class SGC_WFC_WhatsApp_Float {
    const VERSION = '1.0.1';
    const HANDLE_STYLE = 'wfc-whatsapp-style';
    const HANDLE_SCRIPT = 'wfc-whatsapp-inline';

    // Opción 1: Activa debug definiendo en wp-config.php:
    // define('WFC_WHATSAPP_DEBUG', true);
    private static function debug_enabled(): bool {
        return ( defined('WFC_WHATSAPP_DEBUG') && WFC_WHATSAPP_DEBUG )
            || (bool) apply_filters('wfc_whatsapp_debug', false);
    }

    private static function log($message): void {
        if ( ! self::debug_enabled() ) return;
        if ( is_array($message) || is_object($message) ) {
            $message = wp_json_encode($message);
        }
        // Solo servidor (y SOLO si debug está activo)
        error_log('[WFC WhatsApp] ' . $message);
    }

    public static function init(): void {
        add_action('wp_footer', [__CLASS__, 'render_html'], 20);
        add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_assets']);
    }

    private static function should_render(): bool {
        // No mostrar en admin (wp-admin) ni en pantallas de login
        if ( is_admin() ) return false;

        // Evita mostrar a usuarios logueados si quieres (opcional):
        // if ( is_user_logged_in() ) return false;

        return true;
    }

    public static function render_html(): void {
        if ( ! self::should_render() ) return;

        static $printed = false;
        if ( $printed ) return;
        $printed = true;

        // Config actual (puedes luego moverlo a ajustes si quieres)
        $whatsapp_image_url = 'https://premiumhealthsport.com/wp-content/uploads/2025/12/Icon_whatsapp.png';
        $numero_1 = '34915105635';
        $numero_2 = '34916077102';

        if ( empty($whatsapp_image_url) || empty($numero_1) || empty($numero_2) ) {
            self::log('Faltan datos de configuración (imagen o números).');
            return;
        }

        ?>
        <div id="whatsapp-flotante" data-wfc-wa>
            <button id="whatsapp-icono"
                    type="button"
                    aria-expanded="false"
                    aria-controls="whatsapp-popup"
                    aria-label="Abrir opciones de contacto WhatsApp">
                <img src="<?php echo esc_url($whatsapp_image_url); ?>" alt="Contáctanos por WhatsApp" loading="lazy" decoding="async">
            </button>

            <div id="whatsapp-popup" role="dialog" aria-hidden="true" aria-labelledby="popup-title">
                <div class="popup-contenido">
                    <h4 id="popup-title">Selecciona tu ubicación:</h4>

                    <a href="https://wa.me/<?php echo esc_attr($numero_1); ?>"
                       target="_blank"
                       rel="noopener noreferrer"
                       class="popup-btn">
                        Centro de Madrid
                    </a>

                    <a href="https://wa.me/<?php echo esc_attr($numero_2); ?>"
                       target="_blank"
                       rel="noopener noreferrer"
                       class="popup-btn">
                        Centro de Fuenlabrada
                    </a>
                </div>
            </div>
        </div>
        <?php
    }

    public static function enqueue_assets(): void {
        if ( ! self::should_render() ) return;

        // ---------- CSS ----------
        wp_register_style(self::HANDLE_STYLE, false, [], self::VERSION);
        wp_enqueue_style(self::HANDLE_STYLE);

        $css = "
            #whatsapp-flotante{position:fixed;bottom:50px;right:20px;z-index:9999;font-family:-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,Helvetica,Arial,sans-serif}
            #whatsapp-icono{background:#25D366;border:0;border-radius:50%;width:60px;height:60px;padding:0;box-shadow:0 4px 12px rgba(0,0,0,.2);cursor:pointer;transition:transform .2s ease;display:flex;justify-content:center;align-items:center;line-height:1}
            #whatsapp-icono:hover{transform:scale(1.05)}
            #whatsapp-icono img{width:35px;height:35px;display:block}
            #whatsapp-popup{display:none;position:absolute;bottom:75px;right:0;width:250px;max-width:90vw;background:#fff;border-radius:8px;box-shadow:0 4px 15px rgba(0,0,0,.15);padding:15px;opacity:0;transform:translateY(10px);transition:opacity .2s ease,transform .2s ease}
            #whatsapp-popup.visible{display:block;opacity:1;transform:translateY(0)}
            .popup-contenido h4{margin:0 0 15px;font-size:16px;font-weight:700;color:#333;border-bottom:1px solid #eee;padding-bottom:10px}
            .popup-btn{display:block;text-decoration:none;background:#25D366;color:#fff !important;padding:10px 15px;margin-bottom:10px;border-radius:5px;text-align:center;font-size:14px;font-weight:600;transition:background-color .2s ease}
            .popup-btn:last-child{margin-bottom:0}
            .popup-btn:hover{background:#128C7E;color:#fff}
            #whatsapp-popup:after{content:'';position:absolute;bottom:-10px;right:25px;width:0;height:0;border-left:10px solid transparent;border-right:10px solid transparent;border-top:10px solid #fff}
        ";
        wp_add_inline_style(self::HANDLE_STYLE, $css);

        // ---------- JS (inline, sin archivo externo, sin jQuery) ----------
        // Lo registramos como "script vacío" para poder adjuntar inline correctamente
        wp_register_script(self::HANDLE_SCRIPT, '', [], self::VERSION, true);
        wp_enqueue_script(self::HANDLE_SCRIPT);

        $debug = self::debug_enabled() ? 'true' : 'false';

        $js = "
            (function(){
                const DEBUG = {$debug};

                function safeLog(){ if(!DEBUG) return; try{ console.log.apply(console, arguments); }catch(e){} }

                function init(){
                    const flotante = document.querySelector('[data-wfc-wa]');
                    if(!flotante) return;

                    const botonIcono = flotante.querySelector('#whatsapp-icono');
                    const popup = flotante.querySelector('#whatsapp-popup');
                    if(!botonIcono || !popup) return;

                    function setState(open){
                        popup.classList.toggle('visible', open);
                        botonIcono.setAttribute('aria-expanded', open ? 'true' : 'false');
                        popup.setAttribute('aria-hidden', open ? 'false' : 'true');
                    }

                    function toggle(){
                        setState(!popup.classList.contains('visible'));
                    }

                    botonIcono.addEventListener('click', function(e){
                        e.preventDefault();
                        toggle();
                    });

                    document.addEventListener('click', function(e){
                        const isInside = flotante.contains(e.target);
                        if(popup.classList.contains('visible') && !isInside){
                            setState(false);
                        }
                    });

                    document.addEventListener('keydown', function(e){
                        if(e.key === 'Escape' && popup.classList.contains('visible')){
                            setState(false);
                        }
                    });

                    safeLog('[WFC] iniciado');
                }

                if(document.readyState === 'loading'){
                    document.addEventListener('DOMContentLoaded', init);
                }else{
                    init();
                }
            })();
        ";
        wp_add_inline_script(self::HANDLE_SCRIPT, $js, 'after');
    }
}

SGC_WFC_WhatsApp_Float::init();